/**
 * COMPATIBLE READERS: ALL READER PROVIDED WITH USB CONNECTION.
 *
 *   ******  PLEASE, ENSURE THAT READERS like trID or skID are configured in USB BEFORE TESTING
 *           WITH THIS APP (see relatives User/tech manuals for further information)    ******
 *
 * THIS EXAMPLE SHOW A SIMPLE APP THAT FIND THE FIRST USB CAEN RFID DEVICE, TAKE PERMISSION TO
 * COMMUNICATE WITH, AND WAIT FOR INVENTORY. FOR DEMO PURPOSE, THIS APP DOESN'T MANAGE PERMISSION
 * DENIALS, OR DETAILED EXCEPTION HANDLING.
 *
 * THE FLOATING ACTION BUTTON DOES THE FOLLOWING:
 *
 *  1) CONNECT TO THE READER using USB
 *  2) REGISTER EVENT READING  CALLBACK
 *  3) WAIT 3 SECONDS FOR ANY PREVIOUS READING EVENTS.
 *  4) SET POWER TO 50 mW
 *  5) PREPARE CONTINUOUS READING.
 *  6) LAUNCH CONTINUOUS READING.
 *  7) WAITS FOR CONFIGURED SECONDS TO STOP READER.
 *   ... in the meanwhile if any tag is found, then the API will raise the callback...
 *  8) STOP THE CONTINUOUS READING.
 *  9) UNREGISTER THE CALLBACK
 *  10) DISCONNECT FROM USB READER.
 */

package com.example.caenrfidsampleapp;

import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbManager;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.TextView;
import android.widget.Toast;

import androidx.appcompat.app.AppCompatActivity;
import androidx.navigation.NavController;
import androidx.navigation.Navigation;
import androidx.navigation.ui.AppBarConfiguration;
import androidx.navigation.ui.NavigationUI;

import com.caen.RFIDLibrary.CAENRFIDEventListener;
import com.caen.RFIDLibrary.CAENRFIDException;
import com.caen.RFIDLibrary.CAENRFIDLogicalSource;
import com.caen.RFIDLibrary.CAENRFIDNotify;
import com.caen.RFIDLibrary.CAENRFIDReader;
import com.caen.RFIDLibrary.CAENRFIDReaderInfo;
import com.caen.RFIDLibrary.CAENRFIDTag;
import com.caen.VCPSerialPort.VCPSerialPort;
import com.example.caenrfidsampleapp.databinding.ActivityMainBinding;

import java.util.List;
import java.util.concurrent.Semaphore;

public class MainActivity extends AppCompatActivity {

    UsbManager usbManager;
    final Semaphore usbSync = new Semaphore(0, true);
    boolean usbDeviceUsageGranted = false;
    static final String ACTION_USB_PERMISSION = "com.android.example.USB_PERMISSION";

    // The CAEN RFID serial port object that represents the easy2read communication usb interface.
    VCPSerialPort vcpSerialPort = null;

    TextView outputText;

    AppBarConfiguration appBarConfiguration;

    // Broadcast Receiver to catch the permission granting on the USB CAEN RFID Reader.
    final BroadcastReceiver usbReceiver = new BroadcastReceiver() {

        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();
            if (ACTION_USB_PERMISSION.equals(action)) {
                synchronized (this) {
                    UsbDevice device = intent.getParcelableExtra(UsbManager.EXTRA_DEVICE);
                    usbDeviceUsageGranted = intent.getBooleanExtra(UsbManager.EXTRA_PERMISSION_GRANTED, false);
                    if (usbDeviceUsageGranted) {
                        if(device != null){
                            Log.d(getClass().getName(), "permission granted for device " + device);
                        }
                    }
                    else {
                        Log.d(getClass().getName(), "permission denied for device " + device);
                    }
                    usbSync.release();
                }
            }
        }
    };

    // Duration of RFID inventory.
    static final int INVENTORY_DURATION = 5;

    // Utility method to print hex string.
    static final char[] HEX_ARRAY = "0123456789ABCDEF".toCharArray();

    static String bytesToHex(byte[] bytes) {
        if(bytes == null)
            return "NULL";
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; j++) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0x0F];
        }
        return new String(hexChars);
    }

    //CAEN RFID API Callback for EventInventoryTag: it will print the ID part of the EPC of the tag.
    final CAENRFIDEventListener caenrfidEventListener = evt -> {
        CAENRFIDNotify tag = evt.getData().get(0);
        runOnUiThread(() -> this.outputText.append(bytesToHex(tag.getTagID())+"\n"));
    };




    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        usbManager = (UsbManager) getSystemService(Context.USB_SERVICE);
        com.example.caenrfidsampleapp.databinding.ActivityMainBinding binding = ActivityMainBinding.inflate(getLayoutInflater());
        setContentView(binding.getRoot());

        setSupportActionBar(binding.toolbar);

        NavController navController = Navigation.findNavController(this, R.id.nav_host_fragment_content_main);
        appBarConfiguration = new AppBarConfiguration.Builder(navController.getGraph()).build();
        NavigationUI.setupActionBarWithNavController(this, navController, appBarConfiguration);

        IntentFilter filter = new IntentFilter(ACTION_USB_PERMISSION);
        registerReceiver(usbReceiver, filter);

        outputText = binding.getRoot().findViewById(R.id.textview_first);

        //FINDING VCP SERIAL PORTS: use separated Thread!

        new Thread(() -> {
            List<VCPSerialPort> ports = null;
            while (ports == null || ports.isEmpty())
                ports = VCPSerialPort.findVCPDevice(getApplicationContext());
            //NOTE: Some CAEN RFID readers (like TILE) maps the configuration port on usb interface
            // 0, so it's always recommended to use the isEasy2ReadPort() method to identify the
            // correct interface.
            for(VCPSerialPort port : ports) {
                if(port.isEasy2ReadPort()) {
                    vcpSerialPort = port;
                }
            }
            //request permission to use CAEN RFID reader.
            runOnUiThread(() -> {
                PendingIntent permissionIntent = PendingIntent.getBroadcast(getApplicationContext(), 0, new Intent(ACTION_USB_PERMISSION), PendingIntent.FLAG_MUTABLE);
                usbManager.requestPermission(vcpSerialPort.getDriverDevice(),permissionIntent);
                Toast.makeText(getApplicationContext(),"USB Device Found", Toast.LENGTH_SHORT).show();
            });
            // Wait for permission popup decision.
            try {
                usbSync.acquire();
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            //Check decision.
            if(usbDeviceUsageGranted) {
                Log.d(getClass().getName(), "Permission granted");
                runOnUiThread(() -> {
                    Toast.makeText(getApplicationContext(),"USB reader ready, press Play!", Toast.LENGTH_SHORT).show();
                    binding.fab.setEnabled(true);
                });
            } else {
                Log.d(getClass().getName(), "Permission denied");
                Toast.makeText(getApplicationContext(),"Permission to use USB reader denied", Toast.LENGTH_SHORT).show();
                binding.fab.setEnabled(false);
            }
        }).start();
        //
        // FAB Action: Connect, get info, do inventory (continuous mode and single mode) and then
        // disconnect.
        binding.fab.setOnClickListener(view -> {
            //NOTE: Usb device access has been denied, so user should exit from app, detach and
            // reattach device, restart app and accept the permission when required.
            if(!usbDeviceUsageGranted){
                Toast.makeText(getApplicationContext(),"Usb device usage permission has been denied.", Toast.LENGTH_SHORT).show();
                return;
            }
            CAENRFIDReader reader = new CAENRFIDReader();

            // To avoid UI lags, we awaits ten seconds and then stop inventory using a separated
            // Thread. In the meanwhile, any tag identified by the reader will be reported to the
            // caenrfidEventListener callback.
            new Thread(() -> {
                try {
                    runOnUiThread(() -> Toast.makeText(getApplicationContext(),"Inventory Start",Toast.LENGTH_SHORT).show());
                    try {
                        // 1) Connection, setup callback and set 50 mW of conducted power:
                        // vcpSerialPort is the first VCP Serial port device received by the
                        // previous VCP scan operation (see code above).
                        reader.Connect(vcpSerialPort);
                        reader.addCAENRFIDEventListener(caenrfidEventListener);
                        // If a previous EventInventoryTag has been called and reader disconnected
                        // suddenly, then stop the reading. Waiting for data streaming end within 2 s
                        // seconds.
                        boolean endOfStreamFound = reader.ForceAbort(2000);
                        if(endOfStreamFound) {
                            // reader was streaming and therefore in continuous mode.
                        } else {
                            // reader has been not found in continuous mode or maybe timeout is too
                            // short. Typically 2 or 3 seconds is enough to waits the end of
                            // streaming, but it strongly depends on the overall  parameter
                            // settings used during for the inventory reader and the number of
                            // discoverable tags.
                        }
                        //Set 50 mW of conducted power.
                        reader.SetPower(50);
                        //Get reader Info
                        CAENRFIDReaderInfo info = reader.GetReaderInfo();
                        runOnUiThread(() -> outputText.setText(String.format("%s %s\nPut some tags on antenna\n", info.GetModel(), info.GetSerialNumber())));
                        Thread.sleep(1000);
                        // 2) Prepare and launch Event Inventory tag (continuous inventory).
                        reader.GetSource("Source_0").SetReadCycle(0);
                        short flag = (short) (CAENRFIDLogicalSource.InventoryFlag.FRAMED.getValue() + CAENRFIDLogicalSource.InventoryFlag.CONTINUOS.getValue());

                        reader.GetSource("Source_0").EventInventoryTag(new byte[]{},(short)0,(short)0, flag);

                        // This thread waits for INVENTORY_DURATION seconds and in the meanwhile, 
                        // the API will raise an event (CAENRFIDEventListener) for each tag found.
                        Thread.sleep(INVENTORY_DURATION * 1000L);

                        // Finish EnventInventoryTag, try Disconnect and remove event callback.
                        reader.InventoryAbort();
                        // Doing a single inventory
                        Thread.sleep(1000);
                        CAENRFIDTag[] tags = reader.GetSource("Source_0").InventoryTag();
                        if(tags != null) {
                            for (CAENRFIDTag tag :
                                    tags) {
                                runOnUiThread(() -> outputText.append(bytesToHex(tag.GetId())+"\n"));
                            }
                        }
                        reader.Disconnect();
                        reader.removeCAENRFIDEventListener(caenrfidEventListener);
                        runOnUiThread(() -> outputText.append("Disconnected"));
                    } catch (CAENRFIDException e) {
                        //In case of API exception, ensure to remove callback and try to
                        // disconnect again.
                        try {
                            reader.Disconnect();
                        } catch (CAENRFIDException ignored) {}
                        //
                        reader.removeCAENRFIDEventListener(caenrfidEventListener);
                    }
                    runOnUiThread(() -> Toast.makeText(getApplicationContext(),"Inventory Stopped",Toast.LENGTH_SHORT).show());
                } catch (InterruptedException ignored) {}
            }).start();
        });
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.menu_main, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // Handle action bar item clicks here. The action bar will
        // automatically handle clicks on the Home/Up button, so long
        // as you specify a parent activity in AndroidManifest.xml.
        int id = item.getItemId();

        //noinspection SimplifiableIfStatement
        if (id == R.id.action_settings) {
            return true;
        }

        return super.onOptionsItemSelected(item);
    }

    @Override
    public boolean onSupportNavigateUp() {
        NavController navController = Navigation.findNavController(this, R.id.nav_host_fragment_content_main);
        return NavigationUI.navigateUp(navController, appBarConfiguration)
                || super.onSupportNavigateUp();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
    }
}